//
//  vAOTextView.m
//  viAllOver
//
//  Copyright (c) 2004, 2005 Matt O'Brien. All rights reserved.
//
/*
 This file is part of viAllOver.
 
 viAllOver is free software; you can redistribute it and/or modify
 it under the terms of the GNU General Public License as published by
 the Free Software Foundation; either version 2 of the License, or
 (at your option) any later version.
 
 viAllOver is distributed in the hope that it will be useful,
 but WITHOUT ANY WARRANTY; without even the implied warranty of
 MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 GNU General Public License for more details.
 
 You should have received a copy of the GNU General Public License
 along with viAllOver; if not, write to the Free Software
 Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */ 



#import "vAOTextView.h"

@implementation vAOTextView

+ (void)load
{
	// check for apps to exclude
	NSDictionary *excluded = NULL;
	NSDictionary *excluded2 = NULL;
	NSMutableArray *excludeList;
	
	NSBundle * lb = [NSBundle bundleForClass: [vAOData class]];
	
	excluded = [[NSDictionary dictionaryWithContentsOfFile: 
		[lb pathForResource:@"exclude" ofType:@"plist"]] retain];
	
	if ( excluded ) {
		excludeList = [[excluded objectForKey: @"exclude"] retain];
	}
	
	//NSLog(@"trying to exclude list to dict: %@", [lb pathForResource:@"exclude" ofType:@"plist"]);
	
	// now check and see if the user has anything to add
	excluded2 = [[NSDictionary dictionaryWithContentsOfFile: [@"~/Library/Preferences/org.dabble.viAllOver.exclude.plist" stringByExpandingTildeInPath]] retain];
	
	if ( excluded2 ) {
		// we need to add these fancly
		NSArray * newData = [[excluded2 objectForKey: @"exclude"] retain];
		
		[excludeList addObjectsFromArray: newData];
		
		[newData release];
		[excluded2 release];
	}
		
	
	if ( excluded ) {
		//NSLog(@"loaded exclude list");

		NSBundle * b = [NSBundle mainBundle];
		
		NSLog(@"viAllOver is trying to load for %@", [b bundleIdentifier]);
		
		if ( [excludeList containsObject: [b bundleIdentifier]] ) {
			// don't load for this app
			return;
		}
	}
	else {
		NSLog(@"failed loading exclude list");
	}

	[excluded release];	
			
	NSLog(@"loading vAOTextView");
	
	
	// time to take over the world
	[[self class] poseAsClass:[NSTextView class]];		
    
}

- (id)initWithFrame:(NSRect)frameRect
{
	self = [super initWithFrame:frameRect];
	
    if (self) {
        // Initialization code here.
		
		// default to insert mode!
		[[vAOData sharedData] setCommandMode: FALSE];
		[[vAOData sharedData] setRepeatTimes: 0];
		
		[[vAOData sharedData] setInsertionPointLock: FALSE];

	}
	
    return self;
	
}

- (void)keyDown:(NSEvent *)theEvent
{
    // print keyCodes in the textview
    //[super insertText: [[NSNumber numberWithUnsignedShort: [theEvent keyCode]] stringValue]];
	
	BOOL modifiers = FALSE;

    if ( ([theEvent modifierFlags] & NSAlphaShiftKeyMask) ||
		 ([theEvent modifierFlags] & NSShiftKeyMask) ||
		 ([theEvent modifierFlags] & NSControlKeyMask) ||
		 ([theEvent modifierFlags] & NSAlternateKeyMask) ||
		 ([theEvent modifierFlags] & NSCommandKeyMask) ) {
		
		modifiers = TRUE;
	}

    // need to check for escape key here, so we can switch modes
	// [esc] keyCode -> 53
    if ( [theEvent keyCode] == 53 && !modifiers )
    {
		// switch to command mode
		[[vAOData sharedData] setCommandMode: TRUE];
		
		[self setNeedsDisplay:YES];
    }
    else
    {
		[super keyDown: theEvent];
    }
}

- (void)insertText:(id)aString
{    
    if ( [[vAOData sharedData] replaceMode] )
    {
		//NSLog(@"got the replace char");
		
		// get number of times to repeat
		int times = [[vAOData sharedData] repeatTimes];
		int i, rt = (times)?times:1 ;
		
		// now repeat
		for ( i = 0; i < rt; i++ )
		{
			// select the next character to the right
			[self doCommandBySelector: @selector(moveRightAndModifySelection:)];
			
			// replace selected character with new one from user
			[super insertText: aString];
		}
		
		// turn off replace mode
		[[vAOData sharedData] setReplaceMode: FALSE];
		[[vAOData sharedData] setRepeatTimes: 0];
		
		return;
	}
	
	if ( [[vAOData sharedData] commandMode] )
	{	    
	    // check for new lines and forget them
	    if ( [aString isEqualToString: @"\n"] )
	    {
			//[super insertText: aString];
		    return;
	    }

		// check for repeat numbers
	    if ( [aString intValue] > 0 || 
			 ( [[vAOData sharedData] repeatTimes] > 0 && [aString isEqualToString: @"0"]) )
	    {
		    // it's a number, add prevous digits togeather
		    [[vAOData sharedData] setRepeatTimes: 
				(10 * [[vAOData sharedData] repeatTimes]) + [aString intValue]];

		    //NSLog(@"[COMMAND] %d", [[vAOData sharedData] repeatTimes]);

		    return;
	    }
	    
	    // we are always going to repeat at least once
	    if ( [[vAOData sharedData] repeatTimes] < 1 ) {
		    [[vAOData sharedData] setRepeatTimes: 1];
	    }
		
		// try and make a new cammand from the character
		vAOCommand * newCmd = [vAOCommand alloc];
		if (  [newCmd initAndDoItTo: self withChar: aString] ) {
			return;
		}
		[newCmd release];
		
		//NSLog(@"done commanding.");
	    
	    // all done, reset repeatTimes
	    [[vAOData sharedData] setRepeatTimes: 0];
		[[vAOData sharedData] setCommand: NULL];

	}
    else
    {
		// not in command mode, just insert the characters normaly.
		[super insertText: aString];
    }
}

- (void)drawViewBackgroundInRect:(NSRect)aRect
{
	//NSLog(@"drawViewBackgroundInRect: %@", NSStringFromRect(aRect));
	
	[super drawViewBackgroundInRect: aRect];
	
	// now draw our insertion point behind the text
	NSRect caretRect = [self getInsertionPointRect];
	if ( ![self isHidden] && [self isEditable] && NSIntersectsRect(aRect, caretRect) ) {
		[self drawInsertionPointInRect: caretRect color: [self insertionPointColor] turnedOn: TRUE];
	}
}

- (void)updateInsertionPointStateAndRestartTimer:(BOOL)flag {

	if ( [self canDraw] && ![[vAOData sharedData] insertionPointLock]) {		
		
		// we need to have lock because the "textContainerOrigin" method seem to 
		// make "updateInsertionPointStateAndRestartTimer" get called.
		[[vAOData sharedData] setInsertionPointLock: TRUE];
		
		// have the insertion point updated next time
		[self setNeedsDisplayInRect: [self getInsertionPointRect]];
		
		// the following fixed a redraw problem, where the old cursor wasn't getting cleared
		[self setNeedsDisplayInRect: [[vAOData sharedData] oldInsertionPointRect]];
		
		[[vAOData sharedData] setInsertionPointLock: FALSE];
	}
}

- (NSRect)getInsertionPointRect {
	
	unsigned int here = [self selectedRange].location;
	int strLen = [[self string] length];
	
	//NSLog(@"checking here: %d", here);
	
	//NSLog(@"inset: %@", NSStringFromSize([self textContainerInset]));
	//NSLog(@"origin: %@", NSStringFromPoint([self textContainerOrigin]));
	
	NSRect caretRect = NSZeroRect;
	
	if ( strLen == 0 ) {
		// empty textview
		caretRect = [self getStandardInsertionPointRect];
	} else if ( here == strLen ) {
		// when we are at the end of the line.

		here = here - 1;

		//NSLog(@"at the end of the line.");

		if ( [[self string] characterAtIndex: here] == '\n' ) {
			// on a new line
			caretRect = [self getStandardInsertionPointRect];
		} else {

			NSRect charBox = [self rectForCharacterIndex: here];

			// draw the caret in the next slot over
			charBox.origin.x = charBox.origin.x + charBox.size.width;
			charBox.size.width = [self getDefaultCaretWidth];
			
			caretRect = charBox;
		}
		
	} else if ( here >= 0 && here < strLen ) {
		// any where in the middle of the string

		NSRect charBox = [self rectForCharacterIndex: here];
		
		// default if there was a problem
		if ( NSIsEmptyRect(charBox) ) {
			charBox = [[self layoutManager] extraLineFragmentUsedRect];
		} else if (  [[self string] characterAtIndex: here] == '\n' ) {
			// see if prevous char was a newline ask the layout mananger where to draw
			charBox.size.width = [self getDefaultCaretWidth];
		}
		
		caretRect = charBox;
	}
	
	return NSOffsetRect(caretRect, [self textContainerOrigin].x, [self textContainerOrigin].y);
}

- (NSRect)getStandardInsertionPointRect
{
	NSRect r = [[self layoutManager] extraLineFragmentUsedRect];
	
	// get the real insertion point
	r.origin.x = r.origin.x + ( r.size.width / 2 );
	
	r.size.width = [self getDefaultCaretWidth];
	
	return r;
}

- (int)getDefaultCaretWidth
{
	// we need to come up with a way to look this up based one the font.
	// but this is perfect in subethaedit and xcode
	return  6;
}

- (void)drawInsertionPointInRect:(NSRect)aRect color:(NSColor *)aColor turnedOn:(BOOL)flag {	
	// have the old one cleared
	[self setNeedsDisplayInRect: [[vAOData sharedData] oldInsertionPointRect]];
	
	// a nice transparent color
	[[aColor colorWithAlphaComponent:0.40] set];
	//[aColor set];
	
	//NSLog(@"drawing insertion point at: %@", NSStringFromRect(aRect));
	
	// accually draw the caret
	[[NSBezierPath bezierPathWithRect: aRect] fill];
	
	// save the old one so we can make sure it has been cleared
	[[vAOData sharedData] setOldInsertionPointRect: aRect];	

}



- (NSColor *) insertionPointColor {
	
	NSColor * thisColor;
	
	// pick the color
	if ([[self window] firstResponder] != self) { 
		thisColor = [NSColor blackColor]; // inactive
	} else if ( [[vAOData sharedData] commandMode] )
		thisColor = [NSColor redColor]; // command mode
	else
		thisColor = [NSColor greenColor]; // insert mode
	
	return thisColor;
}


/*
 rectForCharacterIndex: method from:
    http://www.cocoabuilder.com/archive/message/cocoa/2005/3/31/131832
    SUBJECT: Re: boundingRectForGlyphRange oddness - what on earth?
    FROM: Keith Blount
    DATE: Thu Mar 31 01:30:08 2005
*/

- (NSRect)rectForCharacterIndex:(int)charIndex
{
	NSLayoutManager *layoutManager = [self layoutManager];
	NSTextContainer *textContainer = [self textContainer];
	unsigned rectCount;
	
	if  ( charIndex < 0 ) 
		return NSZeroRect; 
	
	NSRectArray rectArray = [layoutManager rectArrayForCharacterRange:NSMakeRange(charIndex,1)
		withinSelectedCharacterRange:NSMakeRange(NSNotFound,0) inTextContainer:textContainer rectCount:&rectCount];
	
	if (rectCount)
		return rectArray[0];
	
	return NSZeroRect;
}



// new commands

- (void)vAO_insertMode:(id)sender
{
	[[vAOData sharedData] setCommandMode: FALSE];

    [self setNeedsDisplay:YES];
	
}

- (void)vAO_replaceChar:(id)sender
{
	[[vAOData sharedData] setReplaceMode: TRUE]; 
}

- (void)vAO_undo:(id)sender {
	// don't work, but worth looking into
	//[[self undoManagerForTextView: self] undo];
}

// for implementing "w"
- (void)vAO_wordForward:(id)sender
{
	unsigned int strLength = [[self string] length];
	
	NSRange lookHere = [self selectedRange];
	lookHere.length = strLength - lookHere.location - 1; 
	
	NSMutableCharacterSet * notWordSet = [[NSMutableCharacterSet alloc] init];
	[notWordSet formUnionWithCharacterSet: [NSCharacterSet whitespaceAndNewlineCharacterSet]];
	[notWordSet formUnionWithCharacterSet: [NSCharacterSet punctuationCharacterSet]];
	
	NSMutableCharacterSet * wordSet = [[NSMutableCharacterSet alloc] init];
	[wordSet formUnionWithCharacterSet: [NSCharacterSet alphanumericCharacterSet]];
	[wordSet formUnionWithCharacterSet: [NSCharacterSet punctuationCharacterSet]];
	
	NSRange nextLookHere = [[self string] rangeOfCharacterFromSet: notWordSet options: NSCaseInsensitiveSearch range: lookHere];
	
	if ( nextLookHere.location != NSNotFound ) {
		nextLookHere.length = strLength - nextLookHere.location - 1;
		
		NSRange nextSpot = NSMakeRange(0,0);
		
		do {
			nextSpot = [[self string] rangeOfCharacterFromSet: wordSet options: NSCaseInsensitiveSearch range: nextLookHere];
			nextLookHere.location += 1;
			nextLookHere.length -= 1;
		} while ( nextSpot.location == lookHere.location );
			
		if ( nextSpot.location != NSNotFound ) {
			nextSpot.length = 0;
			[self setSelectedRange: nextSpot];
		}
	}
	
	return;
}

// for implementing "W"
- (void)vAO_wordForwardIgnorePunctuation:(id)sender
{
	unsigned int strLength = [[self string] length];
	
	NSRange lookHere = [self selectedRange];
	lookHere.length = strLength - lookHere.location - 1; 
	
	NSMutableCharacterSet * notWordSet = [[NSMutableCharacterSet alloc] init];
	[notWordSet formUnionWithCharacterSet: [NSCharacterSet whitespaceAndNewlineCharacterSet]];
	[notWordSet formUnionWithCharacterSet: [NSCharacterSet punctuationCharacterSet]];
	
	NSRange nextLookHere = [[self string] rangeOfCharacterFromSet: notWordSet options: NSCaseInsensitiveSearch range: lookHere];
	
	if ( nextLookHere.location != NSNotFound ) {
		nextLookHere.length = strLength - nextLookHere.location - 1;
		
		NSRange nextSpot = [[self string] rangeOfCharacterFromSet: [NSCharacterSet alphanumericCharacterSet] options: NSCaseInsensitiveSearch range: nextLookHere];
		
		if ( nextSpot.location != NSNotFound ) {
			nextSpot.length = 0;
			[self setSelectedRange: nextSpot];
		}
	}
	
	return;
}

@end









